<?php
// File untuk menampilkan grafik langsung tanpa login
require_once 'config/database.php';

$database = new Database();
$db = $database->getConnection();

// Filter periode grafik (default: bulan)
$period = isset($_GET['period']) ? $_GET['period'] : 'month';

// Data grafik pemasangan berdasarkan periode
$chart_data = [];
$chart_labels = [];
$chart_values = [];

if ($period == 'day') {
    // Data per hari (30 hari terakhir)
    $query = "SELECT 
                DATE(created_at) as tanggal,
                COUNT(*) as total
              FROM customers 
              WHERE status = 'installed' 
              AND created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
              GROUP BY DATE(created_at)
              ORDER BY tanggal ASC";
    $stmt = $db->prepare($query);
    $stmt->execute();
    $chart_data = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    foreach ($chart_data as $data) {
        $chart_labels[] = date('d M', strtotime($data['tanggal']));
        $chart_values[] = $data['total'];
    }
} elseif ($period == 'week') {
    // Data per minggu (12 minggu terakhir)
    $query = "SELECT 
                YEARWEEK(created_at, 1) as minggu,
                COUNT(*) as total
              FROM customers 
              WHERE status = 'installed' 
              AND created_at >= DATE_SUB(NOW(), INTERVAL 12 WEEK)
              GROUP BY YEARWEEK(created_at, 1)
              ORDER BY minggu ASC";
    $stmt = $db->prepare($query);
    $stmt->execute();
    $chart_data = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    foreach ($chart_data as $data) {
        $year = substr($data['minggu'], 0, 4);
        $week = substr($data['minggu'], 4, 2);
        $chart_labels[] = "Minggu $week $year";
        $chart_values[] = $data['total'];
    }
} else {
    // Data per bulan (6 bulan terakhir) - default
    $query = "SELECT 
                DATE_FORMAT(created_at, '%Y-%m') as bulan,
                COUNT(*) as total
              FROM customers 
              WHERE status = 'installed' 
              AND created_at >= DATE_SUB(NOW(), INTERVAL 6 MONTH)
              GROUP BY DATE_FORMAT(created_at, '%Y-%m')
              ORDER BY bulan ASC";
    $stmt = $db->prepare($query);
    $stmt->execute();
    $chart_data = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    foreach ($chart_data as $data) {
        $chart_labels[] = date('M Y', strtotime($data['bulan'] . '-01'));
        $chart_values[] = $data['total'];
    }
}
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Grafik Pemasangan - ENSIA</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 2rem 0;
        }
        .chart-container {
            background: white;
            border-radius: 15px;
            padding: 2rem;
            box-shadow: 0 15px 35px rgba(0,0,0,0.1);
            margin: 2rem 0;
        }
        .chart-wrapper {
            position: relative;
            height: 400px;
            width: 100%;
        }
        .header {
            text-align: center;
            color: white;
            margin-bottom: 2rem;
        }
        .header h1 {
            font-size: 2.5rem;
            margin-bottom: 0.5rem;
        }
        .header p {
            font-size: 1.2rem;
            opacity: 0.9;
        }
        .period-btn {
            transition: all 0.3s ease;
            margin: 0.25rem;
        }
        .period-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(0,0,0,0.2);
        }
        .stats-card {
            background: white;
            border-radius: 10px;
            padding: 1.5rem;
            text-align: center;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
            margin-bottom: 1rem;
        }
        .stats-card h3 {
            color: #667eea;
            margin-bottom: 0.5rem;
        }
        .stats-card p {
            color: #666;
            margin: 0;
        }
    </style>
</head>
<body>
    <div class="container">
        <!-- Header -->
        <div class="header">
            <h1><i class="fas fa-wifi me-3"></i>ENSIA</h1>
            <p>Engineer On Site App - PT Maintek Juara Intisolusi</p>
            <h3>📊 Grafik Pemasangan Internet</h3>
        </div>
        
        <!-- Stats Cards -->
        <div class="row mb-4">
            <div class="col-md-3">
                <div class="stats-card">
                    <h3><?php echo array_sum($chart_values); ?></h3>
                    <p>Total Pemasangan</p>
                </div>
            </div>
            <div class="col-md-3">
                <div class="stats-card">
                    <h3><?php echo count($chart_values); ?></h3>
                    <p>Data Points</p>
                </div>
            </div>
            <div class="col-md-3">
                <div class="stats-card">
                    <h3><?php echo $period == 'day' ? '30' : ($period == 'week' ? '12' : '6'); ?></h3>
                    <p><?php echo $period == 'day' ? 'Hari' : ($period == 'week' ? 'Minggu' : 'Bulan'); ?> Terakhir</p>
                </div>
            </div>
            <div class="col-md-3">
                <div class="stats-card">
                    <h3><?php echo count($chart_values) > 0 ? round(array_sum($chart_values) / count($chart_values), 1) : 0; ?></h3>
                    <p>Rata-rata per <?php echo $period == 'day' ? 'Hari' : ($period == 'week' ? 'Minggu' : 'Bulan'); ?></p>
                </div>
            </div>
        </div>
        
        <!-- Grafik -->
        <div class="chart-container">
            <div class="d-flex justify-content-between align-items-center mb-4">
                <h4 class="mb-0">
                    <i class="fas fa-chart-line me-2"></i>
                    Grafik Pemasangan per <?php echo $period == 'day' ? 'Hari' : ($period == 'week' ? 'Minggu' : 'Bulan'); ?>
                </h4>
                <div class="btn-group" role="group">
                    <a href="?period=day" class="btn <?php echo $period == 'day' ? 'btn-primary' : 'btn-outline-primary'; ?> period-btn">
                        <i class="fas fa-calendar-day me-1"></i>Hari
                    </a>
                    <a href="?period=week" class="btn <?php echo $period == 'week' ? 'btn-primary' : 'btn-outline-primary'; ?> period-btn">
                        <i class="fas fa-calendar-week me-1"></i>Minggu
                    </a>
                    <a href="?period=month" class="btn <?php echo $period == 'month' ? 'btn-primary' : 'btn-outline-primary'; ?> period-btn">
                        <i class="fas fa-calendar-alt me-1"></i>Bulan
                    </a>
                </div>
            </div>
            
            <div class="chart-wrapper">
                <canvas id="installationChart"></canvas>
            </div>
            
            <!-- Data Info -->
            <div class="row mt-4">
                <div class="col-md-6">
                    <h6><i class="fas fa-info-circle me-2"></i>Informasi Data:</h6>
                    <ul class="list-unstyled">
                        <li><strong>Periode:</strong> <?php echo $period == 'day' ? '30 hari terakhir' : ($period == 'week' ? '12 minggu terakhir' : '6 bulan terakhir'); ?></li>
                        <li><strong>Total Data:</strong> <?php echo count($chart_values); ?> points</li>
                        <li><strong>Status:</strong> Hanya pelanggan yang sudah diinstalasi</li>
                    </ul>
                </div>
                <div class="col-md-6">
                    <h6><i class="fas fa-chart-bar me-2"></i>Statistik:</h6>
                    <ul class="list-unstyled">
                        <li><strong>Maksimum:</strong> <?php echo count($chart_values) > 0 ? max($chart_values) : 0; ?> pemasangan</li>
                        <li><strong>Minimum:</strong> <?php echo count($chart_values) > 0 ? min($chart_values) : 0; ?> pemasangan</li>
                        <li><strong>Total:</strong> <?php echo array_sum($chart_values); ?> pemasangan</li>
                    </ul>
                </div>
            </div>
        </div>
        
        <!-- Footer -->
        <div class="text-center mt-4">
            <p class="text-white">
                <i class="fas fa-code me-2"></i>
                Developed by Amiruloh untuk Tugas Akhir D3 Manajemen Informatika AMIK CBI Sukabumi
            </p>
            <a href="login.php" class="btn btn-light me-2">
                <i class="fas fa-sign-in-alt me-2"></i>Login ke Aplikasi
            </a>
            <a href="test_chart_periods.html" class="btn btn-outline-light">
                <i class="fas fa-chart-line me-2"></i>Demo Interaktif
            </a>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Grafik pemasangan
        const ctx = document.getElementById('installationChart').getContext('2d');
        
        // Label berdasarkan periode
        const period = '<?php echo $period; ?>';
        let chartLabel = '';
        switch(period) {
            case 'day':
                chartLabel = 'Pemasangan per Hari (30 hari terakhir)';
                break;
            case 'week':
                chartLabel = 'Pemasangan per Minggu (12 minggu terakhir)';
                break;
            case 'month':
            default:
                chartLabel = 'Pemasangan per Bulan (6 bulan terakhir)';
                break;
        }
        
        new Chart(ctx, {
            type: 'line',
            data: {
                labels: <?php echo json_encode($chart_labels); ?>,
                datasets: [{
                    label: chartLabel,
                    data: <?php echo json_encode($chart_values); ?>,
                    borderColor: '#667eea',
                    backgroundColor: 'rgba(102, 126, 234, 0.1)',
                    tension: 0.4,
                    fill: true,
                    pointBackgroundColor: '#667eea',
                    pointBorderColor: '#ffffff',
                    pointBorderWidth: 2,
                    pointRadius: 6,
                    pointHoverRadius: 8
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                aspectRatio: 2.5,
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            stepSize: 1,
                            precision: 0
                        },
                        grid: {
                            color: 'rgba(0,0,0,0.1)'
                        }
                    },
                    x: {
                        grid: {
                            color: 'rgba(0,0,0,0.1)'
                        }
                    }
                },
                plugins: {
                    legend: {
                        display: true,
                        position: 'top'
                    },
                    tooltip: {
                        mode: 'index',
                        intersect: false,
                        backgroundColor: 'rgba(0,0,0,0.8)',
                        titleColor: '#ffffff',
                        bodyColor: '#ffffff',
                        borderColor: '#667eea',
                        borderWidth: 1
                    }
                },
                interaction: {
                    mode: 'nearest',
                    axis: 'x',
                    intersect: false
                }
            }
        });
    </script>
</body>
</html>
