<?php
require_once 'config/config.php';
requireRole(['admin']);

$database = new Database();
$db = $database->getConnection();

// Handle export
if (isset($_GET['export'])) {
    $export_type = $_GET['export'];
    $filter_status = $_GET['filter_status'] ?? '';
    $filter_date_from = $_GET['filter_date_from'] ?? '';
    $filter_date_to = $_GET['filter_date_to'] ?? '';
    $filter_limit = $_GET['filter_limit'] ?? '';
    
    // Build query with filters
    $where_conditions = [];
    $params = [];
    
    if (!empty($filter_status)) {
        $where_conditions[] = "c.status = ?";
        $params[] = $filter_status;
    }
    
    if (!empty($filter_date_from)) {
        $where_conditions[] = "DATE(c.created_at) >= ?";
        $params[] = $filter_date_from;
    }
    
    if (!empty($filter_date_to)) {
        $where_conditions[] = "DATE(c.created_at) <= ?";
        $params[] = $filter_date_to;
    }
    
    $where_clause = !empty($where_conditions) ? 'WHERE ' . implode(' AND ', $where_conditions) : '';
    
    // Add limit based on filter
    $limit_clause = '';
    if (!empty($filter_limit)) {
        switch ($filter_limit) {
            case 'latest_5':
                $limit_clause = 'ORDER BY c.created_at DESC LIMIT 5';
                break;
            case 'cancelled':
                $where_conditions[] = "c.status = 'cancelled'";
                $where_clause = 'WHERE ' . implode(' AND ', $where_conditions);
                break;
            case 'not_installed':
                $where_conditions[] = "c.status IN ('pending', 'approved')";
                $where_clause = 'WHERE ' . implode(' AND ', $where_conditions);
                break;
            case 'installed':
                $where_conditions[] = "c.status = 'installed'";
                $where_clause = 'WHERE ' . implode(' AND ', $where_conditions);
                break;
        }
    }
    
    $query = "SELECT c.*, u.full_name as created_by_name, 
              i.kabel_meter, i.latitude, i.longitude, i.alamat_instalasi, i.catatan, i.tanggal_instalasi,
              u2.full_name as teknisi_name
              FROM customers c 
              LEFT JOIN users u ON c.created_by = u.id 
              LEFT JOIN installations i ON c.id = i.customer_id
              LEFT JOIN users u2 ON i.teknisi_id = u2.id
              $where_clause 
              $limit_clause";
    
    $stmt = $db->prepare($query);
    $stmt->execute($params);
    $customers = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    if ($export_type == 'excel') {
        // Export to Excel
        header('Content-Type: application/vnd.ms-excel');
        header('Content-Disposition: attachment; filename="laporan_pelanggan_' . date('Y-m-d') . '.xls"');
        
        echo "<table border='1'>";
        echo "<tr>";
        echo "<th>No</th>";
        echo "<th>Nama Lengkap</th>";
        echo "<th>Pekerjaan</th>";
        echo "<th>Alamat</th>";
        echo "<th>Nomor HP</th>";
        echo "<th>Email</th>";
        echo "<th>NIK KTP</th>";
        echo "<th>Status</th>";
        echo "<th>Dibuat Oleh</th>";
        echo "<th>Tanggal Input</th>";
        echo "<th>Kabel (Meter)</th>";
        echo "<th>Teknisi</th>";
        echo "<th>Tanggal Instalasi</th>";
        echo "</tr>";
        
        foreach ($customers as $index => $customer) {
            echo "<tr>";
            echo "<td>" . ($index + 1) . "</td>";
            echo "<td>" . $customer['nama_lengkap'] . "</td>";
            echo "<td>" . ($customer['pekerjaan'] ?: '-') . "</td>";
            echo "<td>" . $customer['alamat_lengkap'] . "</td>";
            echo "<td>" . $customer['nomor_hp'] . "</td>";
            echo "<td>" . ($customer['email'] ?: '-') . "</td>";
            echo "<td>" . $customer['nik_ktp'] . "</td>";
            echo "<td>" . ucfirst($customer['status']) . "</td>";
            echo "<td>" . $customer['created_by_name'] . "</td>";
            echo "<td>" . formatDate($customer['created_at']) . "</td>";
            echo "<td>" . ($customer['kabel_meter'] ?: '-') . "</td>";
            echo "<td>" . ($customer['teknisi_name'] ?: '-') . "</td>";
            echo "<td>" . ($customer['tanggal_instalasi'] ? formatDate($customer['tanggal_instalasi']) : '-') . "</td>";
            echo "</tr>";
        }
        echo "</table>";
        exit();
    } elseif ($export_type == 'pdf') {
        // Export to PDF
        require_once 'vendor/autoload.php';
        
        $mpdf = new \Mpdf\Mpdf();
        $mpdf->WriteHTML('
        <h2>Laporan Data Pelanggan</h2>
        <p>Tanggal: ' . date('d/m/Y H:i') . '</p>
        <table border="1" style="border-collapse: collapse; width: 100%;">
        <tr style="background-color: #f2f2f2;">
            <th style="padding: 8px;">No</th>
            <th style="padding: 8px;">Nama Lengkap</th>
            <th style="padding: 8px;">Nomor HP</th>
            <th style="padding: 8px;">Status</th>
            <th style="padding: 8px;">Tanggal Input</th>
            <th style="padding: 8px;">Kabel (Meter)</th>
        </tr>');
        
        foreach ($customers as $index => $customer) {
            $mpdf->WriteHTML('
            <tr>
                <td style="padding: 8px;">' . ($index + 1) . '</td>
                <td style="padding: 8px;">' . $customer['nama_lengkap'] . '</td>
                <td style="padding: 8px;">' . $customer['nomor_hp'] . '</td>
                <td style="padding: 8px;">' . ucfirst($customer['status']) . '</td>
                <td style="padding: 8px;">' . formatDate($customer['created_at']) . '</td>
                <td style="padding: 8px;">' . ($customer['kabel_meter'] ?: '-') . '</td>
            </tr>');
        }
        
        $mpdf->WriteHTML('</table>');
        $mpdf->Output('laporan_pelanggan_' . date('Y-m-d') . '.pdf', 'D');
        exit();
    }
}

// Filter dan pencarian
$search = isset($_GET['search']) ? sanitizeInput($_GET['search']) : '';
$status_filter = isset($_GET['status']) ? sanitizeInput($_GET['status']) : '';
$date_from = isset($_GET['date_from']) ? sanitizeInput($_GET['date_from']) : '';
$date_to = isset($_GET['date_to']) ? sanitizeInput($_GET['date_to']) : '';
$limit_filter = isset($_GET['limit']) ? sanitizeInput($_GET['limit']) : '';

// Query dengan filter
$where_conditions = [];
$params = [];

if (!empty($search)) {
    $where_conditions[] = "(c.nama_lengkap LIKE ? OR c.nomor_hp LIKE ? OR c.email LIKE ? OR c.nik_ktp LIKE ?)";
    $search_param = "%$search%";
    $params = array_merge($params, [$search_param, $search_param, $search_param, $search_param]);
}

if (!empty($status_filter)) {
    $where_conditions[] = "c.status = ?";
    $params[] = $status_filter;
}

if (!empty($date_from)) {
    $where_conditions[] = "DATE(c.created_at) >= ?";
    $params[] = $date_from;
}

if (!empty($date_to)) {
    $where_conditions[] = "DATE(c.created_at) <= ?";
    $params[] = $date_to;
}

$where_clause = !empty($where_conditions) ? 'WHERE ' . implode(' AND ', $where_conditions) : '';

// Query data pelanggan
$query = "SELECT c.*, u.full_name as created_by_name, 
          i.kabel_meter, i.latitude, i.longitude, i.alamat_instalasi, i.catatan, i.tanggal_instalasi,
          u2.full_name as teknisi_name
          FROM customers c 
          LEFT JOIN users u ON c.created_by = u.id 
          LEFT JOIN installations i ON c.id = i.customer_id
          LEFT JOIN users u2 ON i.teknisi_id = u2.id
          $where_clause 
          ORDER BY c.created_at DESC";
$stmt = $db->prepare($query);
$stmt->execute($params);
$customers = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Statistik
$stats_query = "SELECT 
    COUNT(*) as total,
    SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
    SUM(CASE WHEN status = 'approved' THEN 1 ELSE 0 END) as approved,
    SUM(CASE WHEN status = 'installed' THEN 1 ELSE 0 END) as installed,
    SUM(CASE WHEN status = 'cancelled' THEN 1 ELSE 0 END) as cancelled
    FROM customers";
$stats_stmt = $db->prepare($stats_query);
$stats_stmt->execute();
$stats = $stats_stmt->fetch(PDO::FETCH_ASSOC);
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Laporan - <?php echo APP_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .sidebar {
            min-height: 100vh;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }
        .sidebar .logo {
            padding: 1rem;
            text-align: center;
            border-bottom: 1px solid rgba(255,255,255,0.1);
        }
        .sidebar .logo i {
            font-size: 2rem;
            margin-bottom: 0.5rem;
        }
        .sidebar .nav-link {
            color: rgba(255,255,255,0.8);
            padding: 0.75rem 1rem;
            border-radius: 0.375rem;
            margin: 0.25rem 0.5rem;
        }
        .sidebar .nav-link:hover,
        .sidebar .nav-link.active {
            color: white;
            background: rgba(255,255,255,0.1);
        }
        .main-content {
            background: #f8f9fa;
            min-height: 100vh;
        }
        .header {
            background: white;
            padding: 1rem 2rem;
            border-bottom: 1px solid #dee2e6;
            box-shadow: 0 0.125rem 0.25rem rgba(0,0,0,0.075);
        }
        .filter-card {
            background: white;
            border-radius: 0.5rem;
            padding: 1.5rem;
            box-shadow: 0 0.125rem 0.25rem rgba(0,0,0,0.075);
            margin-bottom: 1.5rem;
        }
        .data-card {
            background: white;
            border-radius: 0.5rem;
            box-shadow: 0 0.125rem 0.25rem rgba(0,0,0,0.075);
        }
        .table th {
            border-top: none;
            font-weight: 600;
            color: #495057;
        }
        .export-buttons {
            background: white;
            border-radius: 0.5rem;
            padding: 1rem;
            box-shadow: 0 0.125rem 0.25rem rgba(0,0,0,0.075);
            margin-bottom: 1.5rem;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-3 col-lg-2 px-0">
                <div class="sidebar">
                    <div class="logo">
                        <i class="fas fa-wifi"></i>
                        <h6 class="mb-0"><?php echo APP_NAME; ?></h6>
                        <small><?php echo APP_COMPANY; ?></small>
                    </div>
                    <nav class="nav flex-column mt-3">
                        <a class="nav-link" href="dashboard.php">
                            <i class="fas fa-tachometer-alt me-2"></i>Dashboard
                        </a>
                        <a class="nav-link" href="customers.php">
                            <i class="fas fa-users me-2"></i>Data Pelanggan
                        </a>
                        <a class="nav-link" href="add_customer.php">
                            <i class="fas fa-user-plus me-2"></i>Tambah Pelanggan
                        </a>
                        <a class="nav-link active" href="reports.php">
                            <i class="fas fa-chart-bar me-2"></i>Laporan
                        </a>
                        <a class="nav-link" href="logout.php">
                            <i class="fas fa-sign-out-alt me-2"></i>Logout
                        </a>
                    </nav>
                </div>
            </div>
            
            <!-- Main Content -->
            <div class="col-md-9 col-lg-10 px-0">
                <div class="main-content">
                    <!-- Header -->
                    <div class="header">
                        <div class="d-flex justify-content-between align-items-center">
                            <h4 class="mb-0">Laporan Data Pelanggan</h4>
                            <div class="d-flex align-items-center">
                                <span class="me-3">Selamat datang, <?php echo $_SESSION['full_name']; ?></span>
                                <span class="badge bg-primary"><?php echo ucfirst($_SESSION['user_role']); ?></span>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Content -->
                    <div class="p-4">
                        <!-- Statistik -->
                        <div class="row mb-4">
                            <div class="col-md-3 mb-3">
                                <div class="card text-center">
                                    <div class="card-body">
                                        <h5 class="card-title text-primary"><?php echo $stats['total']; ?></h5>
                                        <p class="card-text">Total Pelanggan</p>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-3 mb-3">
                                <div class="card text-center">
                                    <div class="card-body">
                                        <h5 class="card-title text-warning"><?php echo $stats['pending']; ?></h5>
                                        <p class="card-text">Pending</p>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-3 mb-3">
                                <div class="card text-center">
                                    <div class="card-body">
                                        <h5 class="card-title text-success"><?php echo $stats['installed']; ?></h5>
                                        <p class="card-text">Terinstalasi</p>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-3 mb-3">
                                <div class="card text-center">
                                    <div class="card-body">
                                        <h5 class="card-title text-danger"><?php echo $stats['cancelled']; ?></h5>
                                        <p class="card-text">Dibatalkan</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Export Buttons -->
                        <div class="export-buttons">
                            <h6 class="mb-3">Export Laporan</h6>
                            <div class="row">
                                <div class="col-md-3 mb-2">
                                    <a href="?export=excel&filter_status=<?php echo $status_filter; ?>&filter_date_from=<?php echo $date_from; ?>&filter_date_to=<?php echo $date_to; ?>&filter_limit=<?php echo $limit_filter; ?>" 
                                       class="btn btn-success w-100">
                                        <i class="fas fa-file-excel me-2"></i>Export Excel
                                    </a>
                                </div>
                                <div class="col-md-3 mb-2">
                                    <a href="?export=pdf&filter_status=<?php echo $status_filter; ?>&filter_date_from=<?php echo $date_from; ?>&filter_date_to=<?php echo $date_to; ?>&filter_limit=<?php echo $limit_filter; ?>" 
                                       class="btn btn-danger w-100">
                                        <i class="fas fa-file-pdf me-2"></i>Export PDF
                                    </a>
                                </div>
                                <div class="col-md-3 mb-2">
                                    <a href="?export=excel&filter_limit=latest_5" class="btn btn-outline-success w-100">
                                        <i class="fas fa-download me-2"></i>5 Terbaru
                                    </a>
                                </div>
                                <div class="col-md-3 mb-2">
                                    <a href="?export=excel&filter_limit=installed" class="btn btn-outline-primary w-100">
                                        <i class="fas fa-check-circle me-2"></i>Sudah Instalasi
                                    </a>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Filter -->
                        <div class="filter-card">
                            <form method="GET" class="row g-3">
                                <div class="col-md-4">
                                    <label for="search" class="form-label">Pencarian</label>
                                    <input type="text" class="form-control" id="search" name="search" 
                                           placeholder="Nama, HP, Email, atau NIK..." value="<?php echo $search; ?>">
                                </div>
                                <div class="col-md-2">
                                    <label for="status" class="form-label">Status</label>
                                    <select class="form-select" id="status" name="status">
                                        <option value="">Semua Status</option>
                                        <option value="pending" <?php echo $status_filter == 'pending' ? 'selected' : ''; ?>>Pending</option>
                                        <option value="approved" <?php echo $status_filter == 'approved' ? 'selected' : ''; ?>>Approved</option>
                                        <option value="installed" <?php echo $status_filter == 'installed' ? 'selected' : ''; ?>>Installed</option>
                                        <option value="cancelled" <?php echo $status_filter == 'cancelled' ? 'selected' : ''; ?>>Cancelled</option>
                                        <option value="rejected" <?php echo $status_filter == 'rejected' ? 'selected' : ''; ?>>Rejected</option>
                                    </select>
                                </div>
                                <div class="col-md-2">
                                    <label for="date_from" class="form-label">Dari Tanggal</label>
                                    <input type="date" class="form-control" id="date_from" name="date_from" value="<?php echo $date_from; ?>">
                                </div>
                                <div class="col-md-2">
                                    <label for="date_to" class="form-label">Sampai Tanggal</label>
                                    <input type="date" class="form-control" id="date_to" name="date_to" value="<?php echo $date_to; ?>">
                                </div>
                                <div class="col-md-2 d-flex align-items-end">
                                    <button type="submit" class="btn btn-primary me-2">
                                        <i class="fas fa-search me-2"></i>Cari
                                    </button>
                                    <a href="reports.php" class="btn btn-outline-secondary">
                                        <i class="fas fa-refresh"></i>
                                    </a>
                                </div>
                            </form>
                            
                            <!-- Quick Filters -->
                            <div class="mt-3">
                                <h6>Filter Cepat:</h6>
                                <div class="btn-group" role="group">
                                    <a href="?limit=latest_5" class="btn btn-outline-primary btn-sm">5 Pelanggan Terbaru</a>
                                    <a href="?status=cancelled" class="btn btn-outline-danger btn-sm">Pelanggan Batal</a>
                                    <a href="?status=pending&status=approved" class="btn btn-outline-warning btn-sm">Belum Diinstalasi</a>
                                    <a href="?status=installed" class="btn btn-outline-success btn-sm">Sudah Diinstalasi</a>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Tabel Data -->
                        <div class="data-card">
                            <div class="table-responsive">
                                <table class="table table-hover mb-0">
                                    <thead class="table-light">
                                        <tr>
                                            <th>No</th>
                                            <th>Nama Lengkap</th>
                                            <th>Nomor HP</th>
                                            <th>Email</th>
                                            <th>Status</th>
                                            <th>Dibuat Oleh</th>
                                            <th>Tanggal Input</th>
                                            <th>Kabel (Meter)</th>
                                            <th>Teknisi</th>
                                            <th>Aksi</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php if (empty($customers)): ?>
                                            <tr>
                                                <td colspan="10" class="text-center py-4">
                                                    <i class="fas fa-inbox fa-3x text-muted mb-3"></i>
                                                    <p class="text-muted">Tidak ada data pelanggan</p>
                                                </td>
                                            </tr>
                                        <?php else: ?>
                                            <?php foreach ($customers as $index => $customer): ?>
                                                <tr>
                                                    <td><?php echo $index + 1; ?></td>
                                                    <td>
                                                        <div>
                                                            <strong><?php echo $customer['nama_lengkap']; ?></strong>
                                                            <br>
                                                            <small class="text-muted"><?php echo $customer['pekerjaan']; ?></small>
                                                        </div>
                                                    </td>
                                                    <td><?php echo $customer['nomor_hp']; ?></td>
                                                    <td><?php echo $customer['email'] ?: '-'; ?></td>
                                                    <td>
                                                        <?php
                                                        $status_class = '';
                                                        $status_text = '';
                                                        switch ($customer['status']) {
                                                            case 'pending':
                                                                $status_class = 'warning';
                                                                $status_text = 'Pending';
                                                                break;
                                                            case 'approved':
                                                                $status_class = 'info';
                                                                $status_text = 'Approved';
                                                                break;
                                                            case 'installed':
                                                                $status_class = 'success';
                                                                $status_text = 'Installed';
                                                                break;
                                                            case 'cancelled':
                                                                $status_class = 'danger';
                                                                $status_text = 'Cancelled';
                                                                break;
                                                            case 'rejected':
                                                                $status_class = 'secondary';
                                                                $status_text = 'Rejected';
                                                                break;
                                                        }
                                                        ?>
                                                        <span class="badge bg-<?php echo $status_class; ?>">
                                                            <?php echo $status_text; ?>
                                                        </span>
                                                    </td>
                                                    <td><?php echo $customer['created_by_name']; ?></td>
                                                    <td><?php echo formatDate($customer['created_at']); ?></td>
                                                    <td><?php echo $customer['kabel_meter'] ?: '-'; ?></td>
                                                    <td><?php echo $customer['teknisi_name'] ?: '-'; ?></td>
                                                    <td>
                                                        <div class="btn-group" role="group">
                                                            <a href="customer_detail.php?id=<?php echo $customer['id']; ?>" 
                                                               class="btn btn-sm btn-outline-primary" title="Detail">
                                                                <i class="fas fa-eye"></i>
                                                            </a>
                                                            <a href="customer_print.php?id=<?php echo $customer['id']; ?>" 
                                                               class="btn btn-sm btn-outline-success" title="Print PDF" target="_blank">
                                                                <i class="fas fa-print"></i>
                                                            </a>
                                                        </div>
                                                    </td>
                                                </tr>
                                            <?php endforeach; ?>
                                        <?php endif; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
